﻿using System;
using System.Collections.Generic;
using System.Linq;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;

public partial class Patient : BasePage
{
    private static IReadOnlyList<string> _navigationSources = new List<string>
    {
        "/Patients.aspx",
        "/Referral.aspx",
        "/PatientWorkflows.aspx",
        "/PatientRelatedDiagnoses.aspx",
        "/PatientRelatedLabTests.aspx",
        "/PatientImaging.aspx"
    };

    private readonly EFRPatientManager _manager;
    private PATIENT _patient;

    public Patient()
    {
        _manager = new EFRPatientManager();

        //Manager Save Notification Event Handler
        _manager.BOSaveSuccess += new BaseBO.BOSaveSuccessEvent(manager_BOSaveSuccess);

        NavigationSources = _navigationSources;
    }

    protected void Page_Load(object sender, EventArgs e)
    {
        if (!IsPostBack)
        {
            int patientId;
            if (int.TryParse(Request.QueryString["patientId"], out patientId))
            {
                getRefreshPatient(patientId);

                SetupDemographics();
                BreadCrumbLabel.Text = $"{BreadCrumbLabel.Text}{_patient.LAST_NAME},{_patient.FIRST_NAME}";
                SetPreferredAlternateTelephoneNumber();
            }

            SetReferringPage();
        }
    }

    private void getRefreshPatient(int oId)
    {
        _patient = _manager.SelectByID(oId);
        _patient.SetAsChangeTrackingRoot();
    }

    protected override void LoadViewState(object savedState)
    {
        base.LoadViewState(savedState);
        string patientXml = (string)ViewState["patientXml"];
        _patient = SerializerUtil.Deserialize<PATIENT>(patientXml);

    }

    protected override object SaveViewState()
    {
        ViewState["patientXml"] = _patient.Serialize();
        return base.SaveViewState();

    }

    private void SetupDemographics()
    {
        if (_patient.REGISTRY_DEMOGRAPHICs.Count > 0)
        {
            REGISTRY_DEMOGRAPHIC demo = _patient.REGISTRY_DEMOGRAPHICs.First();

            txtStreetAddress1.Text = demo.ADDRESS_LINE1;
            txtStreetAddress2.Text = demo.ADDRESS_LINE2;
            txtStreetAddress3.Text = demo.ADDRESS_LINE3;
            txtAddressCity.Text = demo.CITY;
            txtAddressState.Text = demo.STATE;
            txtAddressZip.Text = demo.POSTAL_CODE;
            txtAddressCountry.Text = demo.COUNTRY;

            txtHomePhone.Text = demo.HOME_PHONE;
            txtWorkPhone.Text = demo.WORK_PHONE;
            txtMobilePhone.Text = demo.CELL_PHONE;

            txtEmailAddress.Text = demo.EMAIL_ADDRESS;
        }
    }

    private void SetPreferredAlternateTelephoneNumber()
    {
        REGISTRY_DEMOGRAPHIC demo = _patient.REGISTRY_DEMOGRAPHICs.FirstOrDefault();

        if (demo != null)
        {
            if (demo.PREFERRED_PHONE_TYPE == "Home")
            {
                RadioButtonListPreferredAlternateTelephoneNumber.SelectedIndex = 0;
            }
            else if (demo.PREFERRED_PHONE_TYPE == "Work")
            {
                RadioButtonListPreferredAlternateTelephoneNumber.SelectedIndex = 1;
            }
            else if (demo.PREFERRED_PHONE_TYPE == "Mobile")
            {
                RadioButtonListPreferredAlternateTelephoneNumber.SelectedIndex = 2;
            }
        }
    }

    private void SaveCompletePatientDemographics()
    {

        REGISTRY_DEMOGRAPHIC demo;

        if (_patient.REGISTRY_DEMOGRAPHICs.Count > 0)
        {

            demo = _patient.REGISTRY_DEMOGRAPHICs.First();

            SaveDemographics(ref demo);

        }
        else
        {
            demo = new REGISTRY_DEMOGRAPHIC();

            SaveDemographics(ref demo);

            demo.PATIENT_ID = _patient.PATIENT_ID;
            _patient.REGISTRY_DEMOGRAPHICs.Add(demo);
        }
        //set questionnaire in-process status
        _manager.Update(_patient);
        //(this.Master as RegistryMaster).SetDirtyFlag(false);
        var master = this.Master as RegistryMaster;
        if (master != null)
        {
            master.SetDirtyFlag(false);
        }
        getRefreshPatient(_patient.PATIENT_ID);
    }

    private void SaveDemographics(ref REGISTRY_DEMOGRAPHIC rdemo)
    {

        rdemo.ADDRESS_LINE1 = txtStreetAddress1.Text.Trim();
        rdemo.ADDRESS_LINE2 = txtStreetAddress2.Text.Trim();
        rdemo.ADDRESS_LINE3 = txtStreetAddress3.Text.Trim();
        rdemo.CITY = txtAddressCity.Text.Trim();
        rdemo.STATE = txtAddressState.Text.Trim();
        rdemo.POSTAL_CODE = txtAddressZip.Text.Trim();
        rdemo.COUNTRY = txtAddressCountry.Text;

        rdemo.HOME_PHONE = txtHomePhone.Text;
        rdemo.WORK_PHONE = txtWorkPhone.Text;
        rdemo.CELL_PHONE = txtMobilePhone.Text;

        if (RadioButtonListPreferredAlternateTelephoneNumber.SelectedIndex == 0)
        {
            rdemo.PREFERRED_PHONE_TYPE = "Home";
        }
        else if (RadioButtonListPreferredAlternateTelephoneNumber.SelectedIndex == 1)
        {
            rdemo.PREFERRED_PHONE_TYPE = "Work";
        }
        else
        {
            rdemo.PREFERRED_PHONE_TYPE = "Mobile";
        }

        rdemo.EMAIL_ADDRESS = txtEmailAddress.Text;
        //EFR registry
        //TO DO : get registry id from code stored in config
        rdemo.STD_REGISTRY_ID = 1;
        //alt address 
        rdemo.STD_ADDRESSTYPE_ID = 1;
    }

    protected void SaveButton_Click(object sender, EventArgs e)
    {
        SaveCompletePatientDemographics();
        SetupDemographics();
        SetPreferredAlternateTelephoneNumber();
    }

    protected void CancelButton_OnClick(object sender, EventArgs e)
    {
        RedirectToReferringPage();
    }

    public override string TopMenuName()
    {
        return "Patients";
    }

    public override string AppObjectCode()
    {
        return "EFR_PATIENT";
    }

    public override string MenuItemName()
    {
        return "EFR_PATIENT";
    }

}


